<?php
/**
 * A unique identifier is defined to store the options in the database and reference them from the theme.
 * By default it uses the theme name, in lowercase and without spaces, but this can be changed if needed.
 * If the identifier changes, it'll appear as if the options have been reset.
 */

function wcmd_optionsframework_option_name() {
	$wcmd_optionsframework_settings = get_option( 'wcmd_optionsframework' );
	$wcmd_optionsframework_settings['id'] = 'wcmd_options';
	update_option( 'wcmd_optionsframework', $wcmd_optionsframework_settings );
}

/**
  *
  * Add Menu MailChimp Discount
  *
  * @since 1.0
  * @param array $menu
  * @return array $menu
  *
  */
function wcmd_add_settings_menu( $menu ) {
	$menu['page_title']  = 'MailChimp Discount';
	$menu['menu_title']  = 'MailChimp Discount';
	$menu['mode']		 = 'menu';
	$menu['menu_slug']   = 'woocommerce-mailchimp-discount';
	$menu['position']    = '200';
	return $menu;
}
add_filter( 'wcmd_optionsframework_menu', 'wcmd_add_settings_menu' );

/**
  *
  * Get all Shop Coupons
  *
  * @since 1.0
  * @param none
  * @return array id, title
  *
  */
function get_created_coupons() {
	$args = array(
	    'posts_per_page'   => -1,
	    'orderby'          => 'title',
	    'order'            => 'asc',
	    'post_type'        => 'shop_coupon',
	    'post_status'      => 'publish',
	);
    
	$coupons = get_posts( $args );
	$coupon_names = array();
	$coupon_names[''] = __( 'Select Coupon', 'wc_mailchimp_discount' );
	
	foreach ( $coupons as $coupon ) {
	  	$coupon_name = $coupon->post_title;
	  	$coupon_id = $coupon->ID;
	    $coupon_names[$coupon_id] = $coupon_name;
	}
	return $coupon_names;
}

/**
  *
  * Get all User roles
  *
  * @since 1.0
  * @param none
  * @return array
  *
  */
function get_role_names() {
	global $wp_roles;

	$roles = array();

	if ( ! isset( $wp_roles ) )
  		$wp_roles = new WP_Roles();

  	if( is_object($wp_roles) ) {
  		if( isset($wp_roles->roles) ) {
  			$user_roles = $wp_roles->get_names();
  			foreach( $user_roles as $key => $user_role ) {
  				$roles[$key] = $user_role;
  			}
  		}
  	}

  	return $roles;
}

/**
  *
  * Show message for update
  *
  * @since 1.0
  * @param none
  * @return none
  *
  */
function data_import_notices() {

	$screen = get_current_screen();

	if( $screen->id !== 'toplevel_page_woocommerce-mailchimp-discount' )
		return;

	$data_imported = get_option( 'wcmd_data_imported' );

	if( $data_imported == 'yes' )
		return;

	$class = 'notice notice-mg-custom';		
	$message = __( 'We need to update your settings to the latest version of ' . PLUGIN_NAME . '. <a class="wcmd-update-database">Run the Updator</a>', 'wprmenu' );

		printf( '<div class="%1$s"><p>%2$s</p></div>', esc_attr( $class ), $message );
}
add_action( 'admin_notices', 'data_import_notices' );

/**
  *
  * Show message for MailChimp Discount Extra Fields
  *
  * @since 1.0
  * @param none
  * @return none
  *
  */
function show_extra_field_addon_notice() {
	$activated_plugins = get_option( 'active_plugins' );
	$get_site_name = get_site_url();
	$medium = 'addon_notice';
	$extrafields = 'mailchimp-discount-extra-fields-addon';
	$extrafields_slug = $extrafields .'/'. $extrafields . '.php';

  	if( isset( $_GET['page'] ) && $_GET['page'] == 'woocommerce-mailchimp-discount' ) {
    	if( !check_extra_fields_addon_installed() ) {
			$class = 'notice notice-info is-dismissible';   
			$message = __( 'Please install <a href="https://www.magnigenie.com/downloads/woocommerce-mailchimp-discount-extra-fields-add-on?utm_source='.$get_site_name.'&utm_medium='.$medium.'" target="_blank">MailChimp Discount Extra Fields</a> for WooCommerce MailChimp Discount to create your own custom fields and merge them to MailChimp as well', 'wc_mailchimp_discount' );

        	printf( '<div class="%1$s"><p>%2$s</p></div>', esc_attr( $class ), $message );
    	}
  	}
}
add_action( 'admin_notices', 'show_extra_field_addon_notice' );

/**
  *
  * Check MailChimp Discount Extra Fields
  *
  * @since 1.0
  * @param none
  * @return true/false
  *
  */
function check_extra_fields_addon_installed() {
	$activated_plugins = get_option( 'active_plugins' );
	$extrafields = 'mailchimp-discount-extra-fields-addon';
	$extrafields_slug = $extrafields .'/'. $extrafields . '.php';
	$installed = false;

  	if( is_array($activated_plugins) && !empty($activated_plugins) ) {
    	if( in_array($extrafields_slug, $activated_plugins) ) {
      		$installed = true;  
    	}
  	}
  	return $installed;
}

/**
  *
  * get lists
  *
  * @since 1.0
  * @param none
  * @return array
  *
  */
function get_updated_lists() {
  	$get_lists = get_transient( 'wcmd_mailchimp_lists' );

  	if( empty($get_lists) ) {
    	$get_lists = array();
  	}	

  	return $get_lists;
}

/**
  *
  * Get all Pages
  *
  * @since 1.0
  * @param none
  * @return array
  *
  */
function wcmd_get_pages() {
	$pages = get_pages();

	$pages_array = array();
	if( is_array($pages) && !empty($pages) ) {
		foreach( $pages as $page ) {
			$page = (array) $page;
			$pages_array[$page['post_name']] = $page['post_title'];
		}
	}
	return $pages_array;
}

/**
  *
  * Get all Product Category
  *
  * @since 3.6
  * @return array
  *
  */
function wcmd_get_product_cat() {
	$cats_array = array();
	if ( class_exists( 'WooCommerce' ) ) {
		$args = array(
	         'taxonomy'     => 'product_cat',
	         'orderby'      => 'name',
	         'hierarchical' => 1,
	  		);
		$all_categories = get_categories( $args );
		if ( !empty( $all_categories ) ) {
			foreach ( $all_categories as $cat ) {
				$cats_array[$cat->term_id] = $cat->name;
			}
		}
	}
	return $cats_array;
}


/**
  *
  * setup admin options for MailChimp Discount 
  *
  * @since 1.0
  * @param none
  * @return array
  *
  */
$options = get_option( 'wcmd_options' );

function wcmd_optionsframework_options() {

	$get_site_name = get_site_url();
	if ( class_exists( 'WooCommerce' ) ) {
		$cupons_types = wc_get_coupon_types();
	}
	else {
		$cupons_types = '';
	}
	
  	$medium = 'settings_field';

	if ( function_exists( 'icl_object_id' ) )
		$wcmd_email_body = __( 'Email content will be sent to the users when they register on the site. <a href="#" class="wcmd-help">Click here</a> to see the list of variables you can use for <b>Email body and Email subject.</b></br>It Looks like you are using WPML, you can create your preferred language message by <a href="#" class="wcmd-help">Click here</a>', 'wc_mailchimp_discount' );
	else
		$wcmd_email_body = __( 'Email content will be sent to the users when they register on the site. <a href="#" class="wcmd-help">Click here</a> to see the list of variables you can use for <b>Email body and Email subject.</b>', 'wc_mailchimp_discount' );

	$categories = get_terms( 'product_cat', 'orderby=name&hide_empty=0' );
	$cats = array();

	if ( $categories ) {
		foreach ( $categories as $cat ) {
			if ( !empty( $cat->name ) ) {
				$cats[$cat->term_id] = esc_html( $cat->name );
			}
		}
	}
	
	$options = array();


  	$options[] = array( 
  		'name' 		=> __( 'General', 'wc_mailchimp_discount' ),
  		'helper'	=> __( 'Setup Features', 'wc_mailchimp_discount' ),
       	'type' 		=> 'heading',
    );  
		
	$options[] = array( 
		'name' 		=> __( 'Enable', 'wc_mailchimp_discount' ),
		'desc' 		=> __( 'Turn on if you want to enable MailChimp Discount plugin.', 'wc_mailchimp_discount' ),
		'id' 		=> 'wcmd_enabled',
		'std' 		=> '1',
		'type' 		=> 'checkbox',
	);

	$options[] = array( 
		'name' 		=> __( 'Disable Discount', 'wc_mailchimp_discount' ),
		'desc' 		=> __( 'Disable discount for mailchimp sign ups and use it for normal mailchimp signups.', 'wc_mailchimp_discount' ),
		'id' 		=> 'wcmd_disable_discount',
		'std' 		=> '1',
		'type' 		=> 'checkbox',
	);

	$options[] = array( 
		'name' 		=> __( 'Send Coupon To Already Subscribers', 'wc_mailchimp_discount' ),
		'desc' 		=> __( 'Send coupon code to the users who have already subscribed before.', 'wc_mailchimp_discount' ),
		'id' 		=> 'wcmd_send_coupon_subscribed_user',
		'std' 		=> '1',
		'type' 		=> 'checkbox',
	);
	

	$options[] = array( 
		'name' 		=> __( 'Restrict Email', 'wc_mailchimp_discount' ),
		'desc' 		=> __( 'Allow discount if the purchase is made for the same email id user registered on mailchimp.', 'wc_mailchimp_discount' ),
		'id' 		=> 'wcmd_restrict',
		'std'		=> '1',
		'type' 		=> 'checkbox',
	);

	$options[] = array( 
		'name' 		=> __( 'Require User To Be Logged In To Apply Coupon', 'wc_mailchimp_discount' ),
		'desc' 		=>	__( 'If you are using restrict email then you can use this option to require users to be logged in to apply coupon.', 'wc_mailchimp_discount' ),
		'id' 		=>	'wcmd_loggedin',
		'std'		=>	'1',
		'type' 		=>	'checkbox' );

	$options[] = array( 
		'name'		=> __( 'Display Fields', 'wc_mailchimp_discount' ),
    	'id'        =>	'wcmd_fields',
    	'std'       =>	'email_name',
    	'options'   =>	array( 'email' => 'Only Email','email_name' => 'First Name and Email', 'email_name_all' => 'First Name, Last Name and Email' ),
    	'desc'      =>	( !check_extra_fields_addon_installed() ) ? __( 'If you want to create your own custom fields and merge them to mailchimp as well, then please use <a href="https://www.magnigenie.com/downloads/woocommerce-mailchimp-discount-extra-fields-add-on?utm_source='.$get_site_name.'&utm_medium='.$medium.'" target="_blank">WooCommerce Mailchimp Discount - Extra Fields AddOn</a>', 'wc_mailchimp_discount' ) : '',
    	'desc_tip' 	=>	false,
    	'type'     	=> 'select',
    );
	

	$options[] = array( 
		'name' 		=> __( 'Enable Terms and Conditions', 'wc_mailchimp_discount' ),
		'desc' 		=>	__( 'Enable this to use terms and condition in the form', 'wc_mailchimp_discount' ),
		'id' 		=>	'wcmd_terms_condition',
		'std'		=>	'0',
		'type' 		=>	'checkbox',
	);

	$options[] = array( 
		'name' 		=> __( 'Enable Redirect After Successfully Signup', 'wc_mailchimp_discount' ),
		'desc' 		=>	__( 'Enable this for rediect to another page after successful signup', 'wc_mailchimp_discount' ),
		'id' 		=>	'wcmd_signup_redirect',
		'std'		=>	'0',
		'type' 		=>	'checkbox',
	);

	$options[] = array( 
		'name' 		=> __( 'Redirect Url', 'wc_mailchimp_discount' ),
		'desc' 		=>	__( 'Enter Url for which page user needs to be redirect after successfully signup', 'wc_mailchimp_discount' ),
		'id' 		=>	'wcmd_redirect_url',
		'std' 		=>	'https://www.magnigenie.com/shop/',
		'type' 		=>	'text',
	);
	
	$options[] = array( 
		'name' 		=> __( 'Redirect After Seconds', 'wc_mailchimp_discount' ),
		'desc'    	=>	'After how much seconds the page will be redirect?',
		'id' 		=>	'wcmd_redirect_timeout',
		'std' 		=>	'5',
		'class' 	=>	'mini',
		'type' 		=>	'number',
	);

	$options[] = array( 
		'name' 		=> __( 'Terms and Conditions Text', 'wc_mailchimp_discount' ),
		'desc'    	=>	'Terms and Condition text which would be shown in the form',
		'id' 		=>	'wcmd_terms_condition_text',
		'std' 		=>	'Please accept terms and conditions',
		'type' 		=>	'textarea',
		'desc_tip' 	=>	false,
	);

	$options[] = array( 
		'name' 		=> __( 'Terms and Conditions Validation error', 'wc_mailchimp_discount' ),
		'desc'    	=>	'This will be the error text shown when someone doesn\'t make checkbox checked',
		'id' 		=>	'wcmd_terms_condition_error',
		'std' 		=>	'Please accept terms and conditions',
		'type' 		=> 'text',
	);

	$options[] = array( 
		'name' 		=> __( 'Invalid Email Error', 'wc_mailchimp_discount' ),
		'desc'    	=> 'This will be the error text shown when someone enters an invalid email',
		'id' 		=> 'wcmd_email_error',
		'std' 		=> 'Please enter a valid email id',
		'type' 		=> 'text',
	);

	$options[] = array( 
		'name' 		=> __( 'Test E-Mail', 'wc_mailchimp_discount' ),
		'desc'    	=> __( 'This email would be excluded from the MailChimp subscriber list', 'wc_mailchimp_discount' ),
		'id' 		=> 'wcmd_test_mail',
		'std' 		=> get_option( 'admin_email' ),
		'type' 		=> 'text',
	);

	$options[] = array( 
		'name' 		=> __( 'MailChimp Settings', 'wc_mailchimp_discount' ),
  		'helper'	=> __( 'Mailchimp Account Settings', 'wc_mailchimp_discount' ),
       	'type' 		=> 'heading',
    );
	$options[] = array( 
		'name' 		=> __( 'Mailchimp API Key', 'wc_mailchimp_discount' ),
    	'desc'    	=> __( '<a href="http://kb.mailchimp.com/accounts/management/about-api-keys" target="_blank">Get MailChimp API Key</a> | <a class="verify-wcmd-mailchimp-api-key" href="#">Verify MailChimp API Key</a>', 'wc_mailchimp_discount' ),
    	'id'      	=> 'wcmd_api_key',
    	'desc_tip'	=>  false,
    	'std'     	=> '',
    	'type'    	=> 'text',
    );

	$options[] = array( 
		'name' 		=> __( 'Mailchimp List ID/Audience ID', 'wc_mailchimp_discount' ),
    	'desc'    	=>	__( '<a class="wcmd-get-lists" href="#">Get Lists ID/Audience ID</a>', 'wc_mailchimp_discount' ),
    	'id'      	=>	'wcmd_list_id',
    	'options' 	=>	get_updated_lists(),
    	'desc_tip'	=>  false,
    	'std'     	=>	'',
    	'type'    	=>	'select',
    );

	$options[] = array( 
		'name' 		=> __( 'Add Signup Source', 'wc_mailchimp_discount' ),
		'desc' 		=>	__( 'Add SOURCE merge tag for each signup to track the signups', 'wc_mailchimp_discount' ),
		'id' 		=>	'wcmd_source',
		'std' 		=>	'1',
		'type' 		=>	'checkbox',
	);

	$options[] = array( 
		'name'		=> __( 'Signup Source', 'wc_mailchimp_discount' ),
		'desc' 		=>	__( 'This will be the signup source which can be shown in the mailchimp admin to check from where the user has been made signup. Please enter a valid url for the signup source.', 'wc_mailchimp_discount' ),
		'id' 		=>	'wcmd_source_link',
		'std' 		=>	get_home_url(),
		'type' 		=>	'text',
	);

	$options[] = array( 
		'name' 		=> __( 'Double Optin', 'wc_mailchimp_discount' ),
		'desc' 		=>	__( 'This option will make user subscribe to the mailchimp list by doing click on the verification link which would be sent to their email id.', 'wc_mailchimp_discount' ),
		'id' 		=>	'wcmd_double_optin',
		'std' 		=>	'0',
		'desc_tip' 	=>	false,
		'type' 		=>	'checkbox',
	);

	$options[] = array( 
		'name' 		=> __( 'Webhook URL', 'wc_mailchimp_discount' ),
		'desc' 		=> __( 'Enter webhook url. If you want to add this site as your webhook url then add '.site_url().'. You can validate the webhook by doing click here <a class="wcmd_verify_webhook" href="#">here</a>', 'wc_mailchimp_discount' ),
		'id' 		=>	'wcmd_webhook_url',
		'std' 		=>	get_home_url(),
		'desc_tip' 	=>	false,
		'type' 		=>	'text',
	);

	$options[] = array( 
		'name' 		=> __( 'Discount Settings', 'wc_mailchimp_discount' ),
		'helper'	=> __( 'Setup Discount', 'wc_mailchimp_discount' ),
		'type' 		=> 'heading',
	);

	$options[] = array( 
		'name' 		=> __( 'Use Predefined Coupons', 'wc_mailchimp_discount' ),
		'desc' 		=> __( 'Enable this option if you want your users to use the predefined coupons what you have created from the WooCommerce Coupons settings. You can create coupons from  <a href="'.admin_url( 'edit.php?post_type=shop_coupon' ).'" target="_blank">here</a> ', 'wc_mailchimp_discount' ),
		'id' 		=> 'wcmd_use_coupons_from',
		'std' 		=> '0',
		'desc_tip' 	=> false,
		'type' 		=> 'checkbox',
	);

	$options[] = array( 
		'name' 		=> __( 'Select Coupon', 'wc_mailchimp_discount' ),
		'desc' 		=> __( 'Select the coupon for which the code will be sent to the users.', 'wc_mailchimp_discount' ),
		'id' 		=> 'wcmd_predefined_coupon',
		'options' 	=> get_created_coupons(),
		'type' 		=> 'select',
	);

	$options[] = array( 
		'name' 		=> __( 'Select Role', 'wc_mailchimp_discount' ),
		'desc' 		=> __( 'Select for which role you want to assign the coupon', 'wc_mailchimp_discount' ),
		'id' 		=> 'wcmd_roles',
		'class'		=> 'wcmd_multiselect',
		'options' 	=> get_role_names(),
		'type' 		=> 'multiselect',
	);

	$options[] = array( 
		'name' 		=> __( 'Discount Type', 'wc_mailchimp_discount' ),
		'desc' 		=> __( 'Select discount type.', 'wc_mailchimp_discount' ),
		'id' 		=> 'wcmd_dis_type',
		'std' 		=> 'percent',
		'options' 	=> $cupons_types,
		'type' 		=> 'select',
	);

	$options[] = array(
		'name' 		=> __( 'Coupon Prefix', 'wc_mailchimp_discount' ),
		'desc' 		=> __( 'Enter a coupon prefix which would be added before the actual generated coupon code. Leave empty for no prefix.', 'wc_mailchimp_discount' ),
		'id' 		=> 'wcmd_prefix',
		'std' 		=> '',
		'type' 		=> 'text',
	);

	$options[] = array(
		'name' 		=> __( 'Coupon Code Length', 'wc_mailchimp_discount' ),
		'desc' 		=> __( 'Enter a length for the coupon code. Note: the prefix is not counted in coupon code length.', 'wc_mailchimp_discount' ),
		'id' 		=> 'wcmd_code_length',
		'std' 		=> '12',
		'type' 		=> 'number',
	);

	$options[] = array(
		'name' 		=> __( 'Discount Amount', 'wc_mailchimp_discount' ),
		'desc' 		=> __( 'Enter a coupon discount amount', 'wc_mailchimp_discount' ),
		'id' 		=> 'wcmd_amount',
		'std' 		=> '10',
		'type' 		=> 'number',
	);

	$options[] = array( 
		'name' 		=> __( 'Allow Free Shipping', 'wc_mailchimp_discount' ),
		'desc' 		=> __( 'Enable this option if the coupon grants free shipping. The <a href="'.admin_url( 'admin.php?page=wc-settings&amp;tab=shipping&amp;section=WC_Shipping_Free_Shipping' ).'">free shipping method</a> must be enabled with the "must use coupon" setting.', 'wc_mailchimp_discount' ),
		'id' 		=> 'wcmd_shipping',
		'std' 		=> '0',
		'desc_tip' 	=> false,
		'type' 		=> 'checkbox',
	);

	$options[] = array( 
		'name' 		=> __( 'Exclude On Sale Items', 'wc_mailchimp_discount' ),
		'desc' 		=> __( 'Enable this option if the coupon should not apply to items on sale. Per-item coupons will only work if the item is not on sale. Per-cart coupons will only work if there are no sale items in the cart.', 'wc_mailchimp_discount' ),
		'id' 		=> 'wcmd_sale',
		'std' 		=> '0',
		'type' 		=> 'checkbox',
	);

	$options[] = array(
		'name' 		=> __( 'Products', 'wc_mailchimp_discount' ),
		'id' 		=> 'wcmd_products',
		'class' 	=> 'ajax_products',
		'desc' 		=> __( 'Products which need to be in the cart to use this coupon or, for "Product Discounts", which products are discounted.', 'wc_mailchimp_discount' ),
		'type' 		=> 'productselect2',
	);

	$options[] = array(
		'name' 		=> __( 'Exclude Products', 'wc_mailchimp_discount' ),
		'id' 		=> 'wcmd_exclude_products',
		'class' 	=> 'ajax_products',
		'desc' 		=> __( 'Products which must not be in the cart to use this coupon or, for "Product Discounts", which products are not discounted.', 'wc_mailchimp_discount' ),
		'type' 		=> 'productselect2',
	);

	$options[] = array(
		'name' 		=> __( 'Categories', 'wc_mailchimp_discount' ),
		'id' 		=> 'wcmd_categories',
		'class' 	=> 'wcmd_multiselect',
		'options' 	=> $cats,
		'desc' 		=> __( 'A product must be in this category for the coupon to remain valid or, for "Product Discounts", products in these categories will be discounted.', 'wc_mailchimp_discount' ),
		'type' 		=> 'multiselect',
	);

	$options[] = array(
		'name' 		=> __( 'Exclude Categories', 'wc_mailchimp_discount' ),
		'id' 		=> 'wcmd_exclude_categories',
		'class' 	=> 'wcmd_multiselect',
		'options' 	=> $cats,
		'desc' 		=> __( 'Product must not be in this category for the coupon to remain valid or, for "Product Discounts", products in these categories will not be discounted.', 'wc_mailchimp_discount' ),
		'type' 		=> 'multiselect',
	);

	$options[] = array(
		'name' 		=> __( 'Coupon Validity (in days)', 'wc_mailchimp_discount' ),
		'desc' 		=> __( 'Enter number of days the coupon will active from the date of registration of the user. Leave blank for no limit.', 'wc_mailchimp_discount' ),
		'id' 		=> 'wcmd_days',
		'type' 		=> 'number',
	);

	$options[] = array(
		'name' 		=> __( 'Coupon Expiry Date Format', 'wc_mailchimp_discount' ),
		'desc' 		=> __( 'Enter the date format for the coupon expiry date which would be mailed to the user. <a href="http://php.net/manual/en/function.date.php" target="_blank">Click here</a> to know about the available types', 'wc_mailchimp_discount' ),
		'desc_tip' 	=> false,
		'id' 		=> 'wcmd_date_format',
		'std' 		=> 'jS F Y',
		'type' 		=> 'text',
	);

	$options[] = array(
		'name' 		=> __( 'Minimum Purchase', 'wc_mailchimp_discount' ),
		'desc' 		=> __( 'Minimum purchase subtotal in order to be able to use the coupon. Leave blank for no limit', 'wc_mailchimp_discount' ),
		'id' 		=> 'wcmd_min_purchase',
		'std' 		=> '',
		'type' 		=> 'number',
	);

	$options[] = array(
		'name' 		=> __( 'Maximum Purchase', 'wc_mailchimp_discount' ),
		'desc' 		=> __( 'Maximum purchase subtotal in order to be able to use the coupon. Leave blank for no limit', 'wc_mailchimp_discount' ),
		'id' 		=> 'wcmd_max_purchase',
		'std' 		=> '',
		'type' 		=> 'number',
	);

	$options[] = array( 'name' => __( 'Popup Settings', 'wc_mailchimp_discount' ),
		'helper'	=> __( 'Advanced Features', 'wc_mailchimp_discount' ),
		'type' 		=> 'heading' );

	$options[] = array( 
		'name' 		=> __( 'Disable popup', 'wc_mailchimp_discount' ),
		'desc' 		=> __( 'Disable popup and instead you can use [wc_mailchimp_discount] shortcode.', 'wc_mailchimp_discount' ),
		'id' 		=> 'wcmd_disable_popup',
		'std' 		=> '0',
		'type' 		=> 'checkbox',
	);

	$options[] = array( 
		'name' 		=> __( 'Display On Home Only', 'wc_mailchimp_discount' ),
		'desc' 		=>	__( 'Display the popup only when the user visits the homepage.', 'wc_mailchimp_discount' ),
		'id' 		=>	'wcmd_home',
		'std' 		=>	'1',
		'type' 		=>	'checkbox',
	);

	$options[] = array( 
		'name' 		=> __( 'Display Popup On Pages', 'wc_mailchimp_discount' ),
		'desc' 		=>	__( 'Display the popup on selected pages', 'wc_mailchimp_discount' ),
		'id' 		=>	'wcmd_popup_enabled_pages',
		'class' 	=>	'wcmd_multiselect',
		'options' 	=>	wcmd_get_pages(),
		'std' 		=>	'',
		'type' 		=>	'multiselect',
	);

	$options[] = array( 
		'name' 		=> __( 'Display Popup On Products Category', 'wc_mailchimp_discount' ),
		'desc' 		=>	__( 'Display the popup on Products page of selected category.', 'wc_mailchimp_discount' ),
		'id' 		=>	'wcmd_popup_enabled_products',
		'class' 	=>	'wcmd_multiselect',
		'options' 	=>	wcmd_get_product_cat(),
		'std' 		=>	'',
		'type' 		=>	'multiselect',
	);

	$options[] = array( 
		'name' 		=> __( 'Popup Height (px)', 'wc_mailchimp_discount' ),
		'desc' 		=> __( 'Enter a height for the popup. Put 0 for auto height', 'wc_mailchimp_discount' ),
		'id' 		=> 'wcmd_popup_height',
		'std' 		=> '0',
		'type' 		=> 'text',
	);

	$options[] = array( 
		'name' 		=> __( 'Popup Width (px)', 'wc_mailchimp_discount' ),
		'desc' 		=> __( 'Enter a width for the popup. Put 0 for auto width', 'wc_mailchimp_discount' ),
		'id' 		=> 'wcmd_popup_width',
		'std' 		=> '0',
		'desc_tip'	=>  false,
		'type' 		=> 'text',
	);

	$options[] = array( 
		'name' 		=> __( 'Show Popup To Users', 'wc_mailchimp_discount' ),
		'desc' 		=> __( 'Select the popup would be shown to which kind of users?', 'wc_mailchimp_discount' ),
		'id' 		=> 'wcmd_popup_users',
		'options'	=> array( 
			'loggedin_users' 		=> 'Logged in users', 
			'non_loggedin_users' 	=> 'Non logged in users', 
			'all' 					=> 'Show to all',
		),
		'std' 		=> 'all',
		'desc_tip'	=>  true,
		'type' 		=> 'select',
	);

	$options[] = array( 
		'name' 		=> __( 'Button/Link trigger for popup', 'wc_mailchimp_discount' ),
		'desc' 		=> __( 'Use button/link click to trigger the popup. <i>When using this feature make sure your button/link has a class attribute of wcmd-trigger.</i> Example:<code>&lt;a href="#" class="wcmd-trigger"&gt;Open Popup&lt;/a&gt;</code>', 'wc_mailchimp_discount' ),
		'desc_tip'	=> false,
		'id' 		=> 'wcmd_btn_trigger',
		'std' 		=> '0',
		'type' 		=> 'checkbox',
	);

	$options[] = array( 
		'name' 		=> __( 'Open popup Only On Button/Link Click', 'wc_mailchimp_discount' ),
		'desc' 		=> __( 'Enable this option if you want the popup to appear only on button/link click. This will disable automatic popup open feature.', 'wc_mailchimp_discount' ),
		'id' 		=> 'wcmd_only_btn',
		'std' 		=> '0',
		'type' 		=> 'checkbox',
	);

	$options[] = array( 
		'name' 		=> __( 'Exit Intent', 'wc_mailchimp_discount' ),
		'desc' 		=> __( 'Display popup based on exit intent', 'wc_mailchimp_discount' ),
		'id' 		=> 'wcmd_exit_intent',
		'std' 		=> '0',
		'type' 		=> 'checkbox',
	);

	$options[] = array( 
		'name' 		=> __( 'Close Popup On Overlay Click', 'wc_mailchimp_discount' ),
		'desc' 		=> __( 'Close the popup when people click outside the popup?', 'wc_mailchimp_discount' ),
		'id' 		=> 'wcmd_overlay_click',
		'std' 		=> '0',
		'type' 		=> 'checkbox',
	);

	$options[] = array( 
		'name' 		=> __( 'Popup Cookie Length(days)', 'wc_mailchimp_discount' ),
		'desc' 		=> __( 'Enter the value for number of days the site should remember the visitor.', 'wc_mailchimp_discount' ),
		'id' 		=> 'wcmd_cookie_length',
		'std' 		=> '30',
		'type' 		=> 'text',
	);

	$options[] = array( 
		'name' 		=> __( 'Popup Content Top Position (px)', 'wc_mailchimp_discount' ),
		'desc' 		=> __( 'Enter number of pixel for the popup content from the top.', 'wc_mailchimp_discount' ),
		'id' 		=> 'wcmd_content_top',
		'std' 		=> '0',
		'type' 		=> 'number',
	);

	$options[] = array( 
		'name' 		=> __( 'Popup Content Left Position (px)', 'wc_mailchimp_discount' ),
		'desc' 		=> __( 'Enter number of pixel for the popup content from the left. ', 'wc_mailchimp_discount' ),
		'id' 		=> 'wcmd_content_left',
		'std' 		=> '0',
		'type' 		=> 'number',
	);

	$options[] = array( 
		'name' 		=> __( 'Popup Animation effect', 'wc_mailchimp_discount' ),
		'desc' 		=> __( 'Select animation effect for the popup ', 'wc_mailchimp_discount' ),
		'id' 		=> 'wcmd_popup_animation',
		'options' 	=> array(
			'fadeIn' 			=> __( 'Fade In', 'wc_mailchimp_discount' ),
			'fadeInDown' 		=>  __( 'Fade In Down', 'wc_mailchimp_discount' ), 
			'fadeInDownBig' 	=> __( 'Fade In Down Big', 'wc_mailchimp_discount' ), 
			'fadeInLeft' 		=> __( 'Fade In Left', 'wc_mailchimp_discount' ), 
			'fadeInRight' 		=> __( 'Fade In Right', 'wc_mailchimp_discount' ), 
			'fadeInUp' 			=> __( 'Fade In Up', 'wc_mailchimp_discount' ), 
			'flip' 				=> __( 'Flip', 'wc_mailchimp_discount' ), 
			'flipInX' 			=> __( 'Flip In X', 'wc_mailchimp_discount' ), 
			'flipInY' 			=> __( 'Flip In Y', 'wc_mailchimp_discount' ), 
			'lightSpeedIn' 		=> __( 'Light Speed In', 'wc_mailchimp_discount' ), 
			'pulse' 			=> __( 'Pulse', 'wc_mailchimp_discount' ), 
			'rubberBand' 		=> __( 'Rubber Band', 'wc_mailchimp_discount' ), 
			'shake' 			=> __( 'Shake', 'wc_mailchimp_discount' ),
			'tada' 				=> __( 'Tada', 'wc_mailchimp_discount' ),
			'wobble' 			=> __( 'Wobble', 'wc_mailchimp_discount' ), 
			'swing' 			=> __( 'Swing', 'wc_mailchimp_discount' ),
			'bounceIn' 			=> __( 'Bounce In', 'wc_mailchimp_discount' ),
			'bounceInDown' 		=> __( 'Bounce In Down', 'wc_mailchimp_discount' ),
			'bounceInLeft' 		=> __( 'Bounce In Left', 'wc_mailchimp_discount' ),
			'bounceInRight' 	=> __( 'Bounce In Right', 'wc_mailchimp_discount' ),
			'bounceInUp' 		=> __( 'Bounce In Up', 'wc_mailchimp_discount' ),
			'rollIn' 			=> __( 'Roll In', 'wc_mailchimp_discount' ),
			'zoomIn' 			=> __( 'Zoom In', 'wc_mailchimp_discount' ),
			'zoomInDown' 		=> __( 'Zoom In Down', 'wc_mailchimp_discount' ),
			'zoomInLeft' 		=> __( 'Zoom In Left', 'wc_mailchimp_discount' ),
			'zoomInRight' 		=> __( 'Zoom In Right', 'wc_mailchimp_discount' ),
			'zoomInUp' 			=> __( 'Zoom In Up', 'wc_mailchimp_discount' ),
			'rotateIn' 			=> __( 'Rotate In', 'wc_mailchimp_discount' ),
			'rotateInDownLeft' 	=> __( 'Rotate In Down Left', 'wc_mailchimp_discount' ),
			'rotateInDownRight' => __( 'Rotate In Down Right', 'wc_mailchimp_discount' ),
			'rotateInUpLeft' 	=> __( 'Rotate In Up Left', 'wc_mailchimp_discount' ),
			'rotateInUpRight' 	=> __( 'Rotate In Up Right', 'wc_mailchimp_discount' ),
		),
		'std' 		=> 'fadeIn',
		'type' 		=> 'select',
	);

	$options[] = array( 
		'name' 		=> __( 'Close Hinge Effect', 'wc_mailchimp_discount' ),
		'desc' 		=> __( 'Enable hinge effect when closing the modal.', 'wc_mailchimp_discount' ),
		'id' 		=> 'wcmd_hinge',
		'std' 		=> '0',
		'type' 		=> 'checkbox',
	);

	$options[] = array( 
		'name' 		=> __( 'Disable Popup On Mobile', 'wc_mailchimp_discount' ),
		'desc' 		=> __( 'Disable popup on mobile devices', 'wc_mailchimp_discount' ),
		'id' 		=> 'wcmd_disable_mobile_popup',
		'std' 		=> '0',
		'type' 		=> 'checkbox',
	);

	$options[] = array( 
		'name' 		=> __( 'Popup Header Text', 'wc_mailchimp_discount' ),
		'desc' 		=> __( 'Enter the text that would appear on the top of the popup.', 'wc_mailchimp_discount' ),
		'id' 		=> 'wcmd_pop_header',
		'type' 		=> 'textarea',
	);

	$options[] = array( 
		'name' 		=> __( 'Display Form Label', 'wc_mailchimp_discount' ),
		'id' 		=> 'wcmd_form_label',
		'std' 		=> '0',
		'type' 		=> 'checkbox',
	);

	$options[] = array( 
		'name' 		=> __( 'Display Popup After(seconds)', 'wc_mailchimp_discount' ),
		'desc' 		=> __( 'Enter a value for the number of seconds after which the popup will be displayed.', 'wc_mailchimp_discount' ),
		'id' 		=> 'wcmd_dis_seconds',
		'type' 		=> 'number',
	);

	$options[] = array( 
		'name' 		=> __( 'Automatically Close Popup After(seconds)', 'wc_mailchimp_discount' ),
		'desc' 		=> __( 'Enter a value if you want to close the popup automatically when a user successfully subscribes to your mailchimp list. Enter 0 if you don\'t want to close the popup automatically', 'wc_mailchimp_discount' ),
		'id' 		=> 'wcmd_close_seconds',
		'type' 		=> 'number',
	);

	$options[] = array( 
		'name' 		=> __( 'Popup Background Image', 'wc_mailchimp_discount' ),
		'id' 		=> 'wcmd_pop_bg',
		'std' 		=> '',
		'type' 		=> 'upload',
	);

	$options[] = array( 
		'name' 		=> __( 'Popup Text', 'wc_mailchimp_discount' ),
		'desc' 		=> __( 'Popup text will be shown when a new user visits the site. Use <b>{WCMD_FORM}</b> to place the form inside the popup.', 'wc_mailchimp_discount' ),
		'id' 		=> 'wcmd_popup_text',
		'std' 		=> '<p style="text-align: center;"><span style="color: #33d5aa;">ENTER YOUR EMAIL AND GET</span></p><h1 style="text-align: center; margin: 0; font-size: 55px;"><span style="color: #cccccc;">10% OFF</span></h1>{WCMD_FORM}<p style="text-align: center;"><span style="color: #999999;">Be the first to know about our exclusive items, New catalogs and special promotions.</span></p>',
		'desc_tip' 	=> false,
		'type' 		=> 'editor',
	);


	$options[] = array( 
		'name' 		=> __( 'Appearance Settings', 'wc_mailchimp_discount' ),
		'helper'	=> __( 'Customize Looks', 'wc_mailchimp_discount' ),
		'type' 		=> 'heading',
	);

	$options[] = array( 
		'name' 		=> __( 'Popup Background Color', 'wc_mailchimp_discount' ),
		'id' 		=> 'wcmd_pop_bgcolor',
		'std' 		=> '#2b2f3e',
		'type' 		=> 'color',
	);

	$options[] = array( 
		'name' 		=> __( 'Popup Close Button Color', 'wc_mailchimp_discount' ),
		'id' 		=> 'wcmd_close_color',
		'std' 		=> '#FFFFFF',
		'type' 		=> 'color',
	);
	
	$options[] = array( 
		'name' 		=> __( 'Popup Overlay Color', 'wc_mailchimp_discount' ),
		'id' 		=> 'wcmd_pop_overlay',
		'std' 		=> '#2e3865',
		'type' 		=> 'color',
	);

	$options[] = array( 
		'name' 		=> __( 'Popup Terms and Condition Text Color', 'wc_mailchimp_discount' ),
		'id' 		=> 'wcmd_terms_text_color',
		'std' 		=> '#000000',
		'type' 		=> 'color',
	);
	
	$options[] = array( 
		'name' 		=> __( 'Popup Header Text Color', 'wc_mailchimp_discount' ),
		'id' 		=> 'wcmd_header_color',
		'std' 		=> '#000000',
		'type' 		=> 'color',
	);	

	$options[] = array( 
		'name' 		=> __( 'Popup Overlay Opacity', 'wc_mailchimp_discount' ),
		'desc'		=> __( 'Enter a value for the opacity value of the popup background.', 'wc_mailchimp_discount' ),
		'id' 		=> 'wcmd_overlay_opacity',
		'std' 		=> '0.8',
		'type' 		=> 'text',
	);

	$options[] = array( 
		'name' 		=> __( 'Submit Button Text', 'wc_mailchimp_discount' ),
		'std'		=> 'SUBSCRIBE',
		'id' 		=> 'wcmd_btn_text',
		'type' 		=> 'text',
	);

	$options[] = array( 
		'name' 		=> __( 'Submit Button Color', 'wc_mailchimp_discount' ),
		'id' 		=> 'wcmd_btn_color',
		'std' 		=> '#21b990',
		'type' 		=> 'color',
	);	

	$options[] = array( 
		'name' 		=> __( 'Submit Button Hover Color', 'wc_mailchimp_discount' ),
		'id' 		=> 'wcmd_btn_hover',
		'std' 		=> '#21b990',
		'type' 		=> 'color',
	);

	$options[] = array( 
		'name' 		=> __( 'Submit Button Text Color', 'wc_mailchimp_discount' ),
		'id' 		=> 'wcmd_btn_txt_color',
		'std' 		=> '#2b2f3e',
		'type' 		=> 'color',
	);

	$options[] = array( 
		'name' 		=> __( 'Success Message Text Color', 'wc_mailchimp_discount' ),
		'desc'		=> __( 'This will be the text color for the success message', 'wc_mailchimp_discount' ),
		'id' 		=> 'wcmd_success_txt_color',
		'std' 		=> '#21b990',
		'type' 		=> 'color',
	);

	$options[] = array( 
		'name' 		=> __( 'Success Message Background Color', 'wc_mailchimp_discount' ),
		'desc'		=> __( 'This will be the background color for the success message', 'wc_mailchimp_discount' ),
		'id' 		=> 'wcmd_success_bg_color',
		'std' 		=> '#FFFFFF',
		'type' 		=> 'color',
	);

	$options[] = array( 
		'name' 		=> __( 'Error Message Text Color', 'wc_mailchimp_discount' ),
		'desc'		=> __( 'This will be the text color for the error message', 'wc_mailchimp_discount' ),
		'id' 		=> 'wcmd_error_txt_color',
		'std' 		=> '#de0b0b',
		'type' 		=> 'color',
	);

	$options[] = array( 
		'name' 		=> __( 'Error Message Background Color', 'wc_mailchimp_discount' ),
		'desc'		=> __( 'This will be the background color for the error message', 'wc_mailchimp_discount' ),
		'id' 		=> 'wcmd_error_bg_color',
		'std' 		=> '#FFFFFF',
		'type' 		=> 'color',
	);

	$options[] = array( 
		'name' 		=> __( 'Form Width (in px)', 'wc_mailchimp_discount' ),
		'desc'		=> __( 'Enter the subscription form width. Enter 0 for auto width.', 'wc_mailchimp_discount' ),
		'id' 		=> 'wcmd_form_width',
		'std' 		=> '500',
		'type' 		=> 'text',
	);

	$options[] = array( 
		'name' 		=> __( 'Form Alignment', 'wc_mailchimp_discount' ),
		'id' 		=> 'wcmd_form_alignment',
		'options'	=> array( 'left' => 'Left', 'right' => 'Right', 'none' => 'Center' ),
		'std' 		=> 'none',
		'type' 		=> 'select',
	);

	$options[] = array( 
		'name' 		=> __( 'Success Message', 'wc_mailchimp_discount' ),
		'id' 		=> 'wcmd_success_msg',
		'std'		=> __( 'Thank you for subscribing! Check your mail for coupon code!', 'wc_mailchimp_discount' ),
		'desc' 		=> __( 'Enter success message which will appear when user successfully subscribes to your mailchimp list. Use {COUPONCODE} variable for the generated coupon code. Remember this variable would work only in single option', 'wc_mailchimp_discount' ),
		'desc_tip' 	=>	false,
		'type' 		=> 'textarea',
	);

	$options[] = array( 
		'name' 		=> __( 'Email Settings', 'wc_mailchimp_discount' ),
		'helper'	=> __( 'Email Features', 'wc_mailchimp_discount' ),
		'type' 		=> 'heading',
	);

	$options[] = array( 
		'name' 		=> __( 'Email From Name', 'wc_mailchimp_discount' ),
		'desc' 		=> __( 'Enter the name which will appear on the emails.', 'wc_mailchimp_discount' ),
		'id' 			=> 'wcmd_email_name',
		'std' 		=> get_bloginfo( 'name' ),
		'type' 		=> 'text',
	);

	$options[] = array( 
		'name' 		=> __( 'From Email', 'wc_mailchimp_discount' ),
		'desc' 		=> __( 'Enter the email from which the emails will be sent.', 'wc_mailchimp_discount' ),
		'id' 			=> 'wcmd_email_id',
		'std' 		=> get_bloginfo( 'admin_email' ),
		'type' 		=> 'text',
	);

	$options[] = array(
		'name' 		=> __( 'Email Subject', 'wc_mailchimp_discount' ),
		'desc' 		=> __( 'This will be email subject for the emails that will be sent to the users.', 'wc_mailchimp_discount' ),
		'id' 			=> 'wcmd_email_sub',
		'css'    	=> 'width: 100%',
		'std' 		=> 'Congrats, you got a discount for signing up to our Newsletter',
		'type' 		=> 'text',
	);

	$options[] = array(
		'name' 		=> __( 'Email Body', 'wc_mailchimp_discount' ),
		'desc' 		=> __( 'This will be email content for the emails that will be sent to the users.', 'wc_mailchimp_discount' ),
		'id' 			=> 'wcmd_email',
		'std' 		=> '<p>Hi There,</p><p>Thanks for signing up for our Newsletter. As a registration bonus we present you with a 10% of discount on all your orders. The coupon code to redeem the discount is <h3>{COUPONCODE}</h3></p><p>The coupon will expire on {COUPONEXPIRY} so make sure to get the benefits while you still have time.</p><p>Regards</p>',
		'type' 		=> 'editor',
	);

  return $options;
}